# Author: Jens Mattke <jensmattke@aol.com>
# I mainly took the intevation-file and rewrote the functions.

# I tried to make it compatible with three dimensions...

"""Convert Well-Known-Binary format to python objects

Main entry point is the convert_well_known_binary function which takes a
geometry in Well-Known-Binary format and returns a python object with the
geometry.
"""

__version__ = "$Revision: 0.1 $"

from struct import *

# Shape-Types
# the values are those out of the postGIS-system.
_POINT = 1
_LINESTRING = 2
_POLYGON = 3
_MULTIPOINT = 4
_MULTILINESTRING = 5
_MULTIPOLYGON = 6
_GEOMETRYCOLLECTION = 7


def parse_coordinate_lists(wkb,dimension,endian_str):
    """Returns the coordinates in wkb as a list of lists of coordinate pairs/tripples/....

    The wkb parameter is the coordinate part of a geometry in well-known-binary
    format including the number of points.
    The dimension parameter determines the dimension of the coordinates.
    The endian_str parameter gives the correct (endianess-) format-string for
    converting the data.
    """
    geometry = []
    
    # Number of Values in wkb for coordinates
    NoV = unpack( endian_str + "l", wkb[0:4])[0]

    # Does the number of coordinates in the wkb fit the desired dimension?
    if  ( (len( wkb[4:len(wkb)] ) / 8 ) / dimension ) != NoV:
        raise ValueError("Dimension does not fit number of values")
    else:

        # format_string for decoding
        fmt_str = endian_str + ( (NoV * dimension) * "d" )
        
        Data = unpack(fmt_str, wkb[4:len(wkb)])

        geometry = []
        i=0
        poly = []
        while i<len(Data):
            mem = ()
            for j in range(1,dimension+1):
                mem += (Data[i],)
                i+=1
            poly.append((mem[0],mem[1]))
        geometry.append(poly)
        return geometry
    

def parse_multipolygon(wkb,dimension,endian_str):
    """
    Return the MULTIPOLYGON geometry wkb as a list of lists of float pairs
    """
    Info = unpack(endian_str+"Bll",(wkb)[0:9])

    # Does this Multipolygon contain Polygons?
    if Info[1] != _POLYGON:
        raise ValueError("Type-Mismatch in WKB found!")

    # A multipolygon may contain more than one polygon.
    Polygons = []

    # algorithm to parse all the polygons in a multipolygon:
    offset = 9
    for i in range( 1, Info[2]+1):

        # Number of points in the actual segment:
        Points = unpack(endian_str+"l",wkb[offset:offset+4])[0]

        # End of the actual segment
        Ende = offset + (2* Points * 8) + 4

        # Parse one polygon
        Polygons += parse_polygon( wkb[offset:Ende],dimension,endian_str )[0]
        offset = Ende
        
    return [Polygons]
    # This will work properly, but if there are more than one polygon contained,
    # won't thuban draw them connected by a line? That's just false, isn't it?
    # ~~~~JHM

def parse_polygon(wkb,dimension,endian_str):
    """Return the POLYGON geometry in wkt as a list of float pairs"""
    return parse_coordinate_lists(wkb,dimension,endian_str)

def parse_multilinestring(wkb,dimension,endian_str):
    """Return the MULTILINESTRING geometry wkb as a list of lists of float pairs"""
    Info = unpack(endian_str+"Bll",(wkb)[0:9])

    # Does this Multipolygon contain Linestrings?
    if Info[1] != _LINESTRING:
        raise ValueError("Type-Mismatch in WKB found!")

    # A multilinestring may contain more than one linestring.
    Linestrings = []

    offset = 9
    for i in range( 1, Info[2]+1):
        Points = unpack(endian_str+"l",wkb[offset:offset+4])[0]
        Ende = offset + (2* Points * 8) + 4
        Linestrings += parse_linestring( wkb[offset:Ende],dimension,endian_str )[0]
        offset = Ende
        
    return [Linestrings]    
    # I'm not sure if this will work properly. I mean, does thuban know how to
    # separate the linestrings? Same problem as with multipolygons.
    #
    # The same problem would appear with multipoints. Is that the reason
    # thuban doesn't support them with the version of Martin Mueller?
    # ~~~~JHM

def parse_linestring(wkb,dimension,endian_str):
    """Return the LINESTRING geometry in wkb as a list of lists of float pairs"""
    return parse_coordinate_lists(wkb,dimension,endian_str)

def parse_point(wkb,dimension,endian_str):
    """Return the POINT geometry in wkb format as a list of lists of float pairs"""
    return parse_coordinate_lists( pack( endian_str + "l", 1) + wkb,dimension,endian_str)


# map geometry types to parser functions
_function_map = [
    (_MULTIPOLYGON, parse_multipolygon),
    (_POLYGON, parse_polygon),
    (_MULTILINESTRING, parse_multilinestring),
    (_LINESTRING, parse_linestring),
    (_POINT, parse_point),
    ]

def convert_well_known_binary(wkb,dimension):
    """Return the geometry given in well-known-binary format as python objects

    The function accepts only 2D data and supports the POINT, POLYGON,
    MULTIPOLYGON, LINESTRING and MULTILINESTRING geometries.

    The structure of the return value depends on the geometry type. For
    MULTIPOLYGON and MULTILINESTRING return a list of lists of
    coordinate pairs. For POLYGON and LINESTRING return a list of
    coordinate pairs. For POINT return a coordinate pair. All
    coordinates are floats.

    The string wkb may only contain the well-known-binary data of shapes,
    received from a binary cursor of postgreSQL / postGIS
    """

    # little or big endian endcoded wkb ?
    endian_str = "<"
    if unpack("B",(wkb)[0:1])[0] == 0:
        endian_str = ">"

    # parse the Information about the given shape:
    # Info[0] -> endian-flag
    # Info[1] -> shapetype:
    #       1 = POINT
    #       2 = LINESTRING
    #       3 = POLYGON
    #       4 = MULTIPOINT
    #       5 = MULTILINESTRING
    #       6 = MULTIPOLYGON
    #       7 = GEOMETRYCOLLECTION
    #     101 = LINEARRING
    # Info[2] -> number of shapes (should be 1)
    #
    #           Bll = Byte, long, long
    Info = unpack(endian_str+"Bll",(wkb)[0:9])

    # Shape-Type
    shp_typ = Info[1]
    
    for geotype, function in _function_map:
        if shp_typ == geotype:
            if shp_typ == _POINT:
                return function(wkb[5:len(wkb)],dimension,endian_str)
            return function(wkb[9:len(wkb)],dimension,endian_str)
    else:
        raise ValueError("Unsupported WKB found!")
    

def parse_wkb_thuban(wkb,dimension):
    """Like convert_well_known_text"""
    return convert_well_known_binary(wkb,dimension)

# I didn't find the difference between 'parse_wkt_thuban()' and
# 'convert_well_known_text()'. So in my version the second just
# calls the other one.

